package de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets;

import java.io.IOException;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.eclipse.jetty.security.authentication.SessionAuthentication;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.GeographicRegionDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.certificates.datacontainers.PsidSspPriorityDataContainer;
import de.fraunhofer.sit.c2x.pki.ca.core.exceptions.HandlerException;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.HtmlProvider;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.servlets.interfaces.AuthenticatorWebHandler;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class AuthenticatorRegistrationServlet extends AbstractServlet {

	private static final long serialVersionUID = 1L;
	public static final String URL = "/registerAuthenticator";
	public static final String TITLE = "Register authenticator";
	private String info;
	private static final int NUM_PERMISSIONS = 5;

	public static final Pattern VALID_EMAIL_ADDRESS_REGEX = Pattern.compile(
			"^[A-Z0-9._%+-]+@[A-Z0-9.-]+\\.[A-Z]{2,6}$", Pattern.CASE_INSENSITIVE);

	public static boolean validate(String emailStr) {
		Matcher matcher = VALID_EMAIL_ADDRESS_REGEX.matcher(emailStr);
		return matcher.find();
	}

	@Inject
	private AuthenticatorWebHandler authenticatorWebHandler;

	@Inject
	public AuthenticatorRegistrationServlet(HtmlProvider htmlProvider) {
		super(htmlProvider);
	}

	public static String safeToString(Object o) {
		if (o == null) {
			return "";
		}
		return o.toString();
	}

	@Override
	public String getUrl() {
		return URL;
	}

	@Override
	public String getTitle() {
		return TITLE;
	}

	@Override
	protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException,
			IOException {
		doGet(req, resp);
	}

	@Override
	protected String contentHtml(HttpServletRequest req, HttpServletResponse resp) {

		StringBuilder sb = new StringBuilder();
		info = "";

		if (req.getParameter("email") != null && req.getParameter("email").length() > 0
				&& req.getParameter("company") != null && req.getParameter("company").length() > 0
				&& req.getParameter("name") != null && req.getParameter("name").length() > 0
				&& req.getParameter("certificate") != null && req.getParameter("certificate").length() > 0
				&& req.getParameter("assuranceLevel") != null
				&& req.getParameter("assuranceLevel").length() > 0) {

			if (logger.isDebugEnabled())
				logger.debug("All required parameters given");

			String company = req.getParameter("company");
			String name = req.getParameter("name");
			String email = req.getParameter("email");
			String telephone = req.getParameter("telephone");
			if (telephone == null)
				telephone = "";
			String address = req.getParameter("address");
			if (address == null)
				address = "";
			if (validate(email) == false) {
				if (info.isEmpty())
					info = "<div id=\"errorbox\">Invalid email address</div>";
				if (logger.isDebugEnabled())
					logger.debug("Invalid email address");
			}

			// read and check assurance level
			if (req.getParameter("assuranceLevel").length() != 2) {
				if (info.isEmpty())
					info = "<div id=\"errorbox\">Assurance Level as byte required</div>";
				if (logger.isDebugEnabled())
					logger.debug("Assurance Level as byte required.");
			}
			byte[] assuranceLevel = null;
			try {
				assuranceLevel = Hex.decodeHex(req.getParameter("assuranceLevel").toCharArray());
			} catch (DecoderException e) {
				if (info.isEmpty())
					info = "<div id=\"errorbox\">Given assurance Level " + req.getParameter("assuranceLevel")
							+ " (HEX) cannot be decoded</div>";
				if (logger.isDebugEnabled())
					logger.debug("Given assurance Level " + req.getParameter("assuranceLevel")
							+ " (HEX) cannot be decoded");
			}

			// read certificate
			String certificateString = req.getParameter("certificate");

			// read permissions if available
			ArrayList<PsidSspPriorityDataContainer> permissions = new ArrayList<PsidSspPriorityDataContainer>(
					NUM_PERMISSIONS);
			for (int i = 1; i <= NUM_PERMISSIONS; i++) {
				PsidSspPriorityDataContainer permissionContainer = null;
				try {
					if (req.getParameter("aid" + i) != null && req.getParameter("aid" + i).length() > 0) {
						permissionContainer = new PsidSspPriorityDataContainer();
						long psid = new Long(req.getParameter("aid" + i));
						permissionContainer.setPsid(psid);
						if (req.getParameter("ssp" + i) != null && req.getParameter("ssp" + i).length() > 0) {
							permissionContainer.setServiceSpecificPermissions(req.getParameter("ssp" + i));
						}
						if (req.getParameter("priority" + i) != null
								&& req.getParameter("priority" + i).length() > 0) {
							int priority = new Integer(req.getParameter("priority" + i));
							permissionContainer.setMaxPriority(priority);
						}
					}
				} catch (NumberFormatException | DecoderException e) {
					if (info.isEmpty())
						info = "<div id=\"errorbox\">" + i + ". Permission cannot be  read: "
								+ e.getMessage() + "</div>";
					if (logger.isDebugEnabled())
						logger.debug(i + ". Permission cannot be read: " + e.getMessage());
				}
				if (permissionContainer != null)
					permissions.add(permissionContainer);
			}
			// read regions
			GeographicRegionDataContainer region = null;
			if (req.getParameter("regionDictionary") != null
					&& req.getParameter("regionDictionary").length() > 0) {
				String regionIdString = req.getParameter("regionId");
				String regionLocalIdString = req.getParameter("regionLocalId");
				String regionDictionaryString = req.getParameter("regionLocalId");
				if (regionIdString.isEmpty()) {
					if (logger.isDebugEnabled())
						logger.debug("Region ID cannot be read: " + regionIdString);
					if (info.isEmpty())
						info = "<div id=\"errorbox\">Region ID cannot be read</div>";
				} else if (regionLocalIdString.isEmpty()) {
					if (logger.isDebugEnabled())
						logger.debug("Region local ID cannot be read: " + regionLocalIdString);
					if (info.isEmpty())
						info = "<div id=\"errorbox\">Region local ID cannot be read</div>";
				} else if (regionDictionaryString.isEmpty()) {
					if (logger.isDebugEnabled())
						logger.debug("Region dictionary cannot be read: " + regionDictionaryString);
					if (info.isEmpty())
						info = "<div id=\"errorbox\">Region dictionary cannot be read</div>";
				} else {
					Long regionId = new Long(regionIdString);
					Long regionLocalId = new Long(regionLocalIdString);
					region = new GeographicRegionDataContainer(req.getParameter("regionDictionary"),
							regionId, regionLocalId);
				}
			} else if (req.getParameter("regionCircleLatitude") != null
					&& req.getParameter("regionCircleLatitude").length() > 0
					&& req.getParameter("regionCircleLongitude") != null
					&& req.getParameter("regionCircleLongitude").length() > 0
					&& req.getParameter("regionCircleRadius") != null
					&& req.getParameter("regionCircleRadius").length() > 0) {
				Double centerLatitude = new Double(req.getParameter("regionCircleLatitude"));
				Double centerLongitude = new Double(req.getParameter("regionCircleLongitude"));
				Integer radius = new Integer(req.getParameter("regionCircleRadius"));
				region = new GeographicRegionDataContainer(centerLatitude, centerLongitude, radius);
			}

			// insert data into db
			if (info.isEmpty()) {
				try {

					if (logger != null && logger.isDebugEnabled()) {
						logger.debug("START: handling of an authenticator registration");
					}

					// get user from http session
					SessionAuthentication user = (SessionAuthentication) req.getSession().getAttribute(
							SessionAuthentication.__J_AUTHENTICATED);

					boolean res = authenticatorWebHandler.handleAuthenticatorRegistration(company, name,
							email, telephone, address, assuranceLevel, certificateString, permissions,
							region, user.getUserIdentity().getUserPrincipal().getName().toString());

					if (logger.isInfoEnabled())
						logger.info("USER: User "
								+ user.getUserIdentity().getUserPrincipal().getName().toString()
								+ " registered a new authenticator with email address: " + email);

					if (res) {
						info = "<div id=\"ackbox\">Authenticator \"" + email + "\" stored in DB</div>";

						if (logger != null && logger.isDebugEnabled()) {
							logger.debug("Authenticator \"" + email + "\" stored in DB");
							logger.debug("END: handling of an authenticator registration (successful)");
						}
					}
				} catch (HandlerException e) {
					if (info.isEmpty())
						info = "<div id=\"errorbox\">Authenticator \"" + email + "\" not stored in DB: "
								+ e.getMessage() + "</div>";

					if (logger != null && logger.isDebugEnabled()) {
						logger.debug("Authenticator \"" + email + "\" not stored in DB", e);
						logger.debug("END: handling of an authenticator registration (failed)");
					}
				}
			}
		} else if (req.getParameter("submit") != null) {
			if (info.isEmpty())
				info = "<div id=\"errorbox\">Please provide all required elements</div>";
		}

		sb.append("<form action=\"" + URL + "\" method=\"get\">\n");
		if (!info.isEmpty())
			sb.append(info);
		sb.append("<table border=\"0\">\n");
		sb.append("<tr><td>Company*:</td><td><input id=\"company\" name=\"company\" type=\"text\" value=\"\" style=\"width:600px\" maxlength=\"50\" /></td></tr>\n");
		sb.append("<tr><td>Name*:</td><td><input id=\"name\" name=\"name\" type=\"text\" value=\"\" style=\"width:600px\" maxlength=\"50\" /></td></tr>\n");
		sb.append("<tr><td>Email address*:</td><td><input id=\"email\" name=\"email\" type=\"text\" value=\"\" style=\"width:600px\" maxlength=\"70\" /></td></tr>\n");
		sb.append("<tr><td>Telephone:</td><td><input id=\"telephone\" name=\"telephone\" type=\"text\" value=\"\" style=\"width:600px\" maxlength=\"30\" /></td></tr>\n");
		sb.append("<tr><td>Address:</td><td><textarea id=\"address\" name=\"address\" rows=\"2\" style=\"width:600px\"></textarea></td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>X.509 certificate*:</td><td><textarea id=\"certificate\" name=\"certificate\" rows=\"10\" style=\"width:600px\"></textarea></td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Assurance level*:</td><td>\n");
		sb.append("<select id=\"assuranceLevel\" name=\"assuranceLevel\" style=\"width:600px\">");
		sb.append("<option value=\"00\">0</option>\n");
		sb.append("<option value=\"20\">1</option>\n");
		sb.append("<option value=\"40\">2</option>\n");
		sb.append("<option value=\"60\">3</option>\n");
		sb.append("<option value=\"80\">4</option>\n");
		sb.append("<option value=\"A0\">5</option>\n");
		sb.append("<option value=\"C0\">6</option>\n");
		sb.append("<option value=\"E0\">7</option>\n");
		sb.append("</select></td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Permissions:</td><td>");
		sb.append("<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">");
		for (int i = 1; i <= NUM_PERMISSIONS; i++) {
			sb.append("<tr>\n");
			sb.append("<td style=\"width:200px\" align=\"left\">" + i + ". AID (int):&nbsp;<input id=\"aid"
					+ i + "\" name=\"aid" + i
					+ "\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"8\" /></td>");
			sb.append("<td style=\"width:200px\" align=\"center\">" + i + ". SSP (hex):&nbsp;<input id=\"ssp"
					+ i + "\" name=\"ssp" + i
					+ "\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"4\" /></td>");
			sb.append("<td style=\"width:200px\" align=\"right\">" + i
					+ ". Priority (int):&nbsp;<input id=\"priority" + i + "\" name=\"priority" + i
					+ "\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"8\" /></td>");
			sb.append("</tr>");
		}
		sb.append("</table>\n");
		sb.append("</td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td>Region:</td><td>");
		sb.append("<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">");
		sb.append("<tr>\n");
		sb.append("<td style=\"width:100px\" align=\"left\">IdentifiedRegion:&nbsp;</td>");
		sb.append("<td style=\"width:170px\" align=\"right\">Dictionary:&nbsp;<select id=\"regionDictionary\" name=\"regionDictionary\" style=\"width:100px\"><option value=\"\"></option><option value=\"iso_3166_1\">iso_3166_1</option><option value=\"un_stats\">un_stats</option></select></td>");
		sb.append("<td style=\"width:170px\" align=\"right\">Region ID (int16):&nbsp;<input id=\"regionId\" name=\"regionId\" type=\"text\" value=\"\" style=\"width:50px\" maxlength=\"5\" /></td>");
		sb.append("<td style=\"width:160px\" align=\"right\">Local ID (int16):&nbsp;<input id=\"regionLocalId\" name=\"regionLocalId\" type=\"text\" value=\"0\" style=\"width:50px\" maxlength=\"5\" /></td>");
		sb.append("</tr>");
		sb.append("<tr>\n");
		sb.append("<td align=\"left\">CircularRegion:&nbsp;</td>");
		sb.append("<td align=\"right\">Lat. (int):&nbsp;<input id=\"regionCircleLatitude\" name=\"regionCircleLatitude\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"11\" /></td>");
		sb.append("<td align=\"right\">Long. (int):&nbsp;<input id=\"regionCircleLongitude\" name=\"regionCircleLongitude\" type=\"text\" value=\"\" style=\"width:100px\" maxlength=\"11\" /></td>");
		sb.append("<td align=\"right\">Radius (uint16):&nbsp;<input id=\"regionCircleRadius\" name=\"regionCircleRadius\" type=\"text\" value=\"\" style=\"width:50px\" maxlength=\"5\" /></td>");
		sb.append("</tr>");
		sb.append("</table>\n");
		sb.append("</td></tr>\n");

		sb.append("<tr><td></td><td>&nbsp;</td></tr>\n");
		sb.append("<tr><td></td><td><input type=\"submit\" value=\"Register authenticator\" /></td></tr>\n");
		sb.append("</table>\n" + "</form>\n");
		sb.append("<br/>\nRequired elements are marked by a *\n");
		sb.append("<br/>\n");
		sb.append("<br/>\n");

		sb.append(getUserInfo());

		return sb.toString();
	}

	private String getUserInfo() {
		StringBuilder sb = new StringBuilder();
		sb.append("<hr width=\"100%\" style=\"size: 1px; color: #000000;\" />");
		sb.append("<h1>Details about form elements</h1>\n");
		sb.append("<ul>\n");
		sb.append("<li>The organization in the Distinguished Name (DN) of the X.509v3 certificate has to be the same as the company of the user.</li>\n");
		sb.append("<li>The email address in the Distinguished Name (DN) of the X.509v3 certificate has to be the same as the email address of the user.</li>\n");
		sb.append("<li>A single PEM (Privacy Enhanced Mail) Base64 encoded DER X.509v3 certificate, enclosed between \"-----BEGIN CERTIFICATE-----\" and \"-----END CERTIFICATE-----\" is required.</li>\n");
		sb.append("<li>Assurance level as unsigned integer in the range [0,7] according to ETSI 103 097 v1.1.1</li>\n");
		sb.append("<li>List of Permissions can be empty or partly be filled. If an SSP or Priority is given in a row then the corresponding ITS-AID must be given in the same row.\n");
		sb.append("<ul>\n");
		sb.append("<li>ITS Application ID (ITS-AID) as unsigned integer in the range [0,2^16-1] according to ETSI 103 097 v1.1.1</li>\n");
		sb.append("<li>Service Specific Permission (SSP) as hex encoded string with a maximum size of 8 characters which corresponds to 4 Bytes. The SSP is interpreted as bit array. The semantic of the bits is not further specified.</li>\n");
		sb.append("<li>Priority as unsigned integer in the range [0,2^8-1] according to ETSI 103 097 v1.1.1</li>\n");
		sb.append("</ul>\n");
		sb.append("</li>");
		sb.append("<li>Both entries of the Region can be empty. If an Identified Region is given then no Circular Region must be added and vice versa.\n");
		sb.append("<ul>\n");
		sb.append("<li>The Dictionary of the Identified Region must be unsigned integer in the range [0,2^8-1] according to ETSI 103 097 v1.1.1 but can simply be selected from the drop down list.</li>\n");
		sb.append("<li>If a Dictionary is selected then the Region ID has to be given as unsigned integer in the range [0,2^16-1] according to ETSI 103 097 v1.1.1</li>\n");
		sb.append("<li>The Local ID must be set if a Dictionary and Region ID is given. The Local ID must be unsigned integer in the range [0,2^32-1] according to ETSI 103 097 v1.1.1. The value 0 is the default value and indicates that no specific local identifiert is used.</li>\n");
		sb.append("<li>If the Circular Region is added then all three elements Latitude, Longitude and Region have to be given. The permitted values of Latitude range from  900 000 000 to +900 000 000 according to ETSI 103 097 v1.1.1. The value 900 000 001 shall indicate the latitude as not being available. The permitted values of Longitude range from -1 800 000 000 to +1 800 000 000. The value 1800 000 001 shall indicate the longitude as not being available. The Redius must be given as unsigned integer in the range [0,2^16-1] according to ETSI 103 097 v1.1.1</li>\n");
		sb.append("</ul>\n");
		sb.append("</li>");
		sb.append("</ul>\n");
		sb.append("<br/>\n");
		sb.append("<hr width=\"100%\" style=\"size: 1px; color: #000000;\" />");
		sb.append("<h1>How to generate a self-signed X.509v3 certificate</h1>\n");
		sb.append("<h2>Preparations</h2>\n");
		sb.append("<ul>\n");
		sb.append("<li>Install OpenSSL (Windows users can get binaries here: <a href=\"http://slproweb.com/products/Win32OpenSSL.html\">http://slproweb.com/products/Win32OpenSSL.html</a>)</li>\n");
		sb.append("<li>Add the OpenSSL bin directory to your system path (reboot might be required)</li>\n");
		sb.append("<li>Open a command line</li>\n");
		sb.append("</ul>\n");
		sb.append("<h2>Create a new self-signed X509 certificate using openSSL</h2>\n");
		sb.append("<p>Generate a key in the command line: <pre>openssl genrsa -des3 -out mycertificate.key 2048</pre></p>\n");
		sb.append("<p>Enter a password for the private key and repeat it when asked.</p>\n");
		sb.append("<p>Generate the certificate: <pre>openssl req -new -x509 -days 3650 -key mycertificate.key -out mycertificate.cert</pre></p>\n");
		sb.append("<p>In this case, the certificate has a validity of 10 years (3650 days). \"mycertificate.key\" contains your private key and \"mycertificate.cert\" is the actual public certificate.</p>\n");
		sb.append("<p>After running the command you will be asked for several details like your company name and email address. Please fill out these fields as good as possible. Especially the email address field has to be completed and should contain the same email address as used in the registration.</p>\n");
		sb.append("<p>The resulting X.509v3 certificate \"mycertificate.cert\" can be opened in a text editor (e.g. notepad++, wordpad or textpad) and should contain at the beginning the text \"-----BEGIN CERTIFICATE-----\" and at the end \"-----END CERTIFICATE-----\". The complete text must be copied into the registration form of this document or into the authenticator registration web page form of the LTCA.</p>\n");
		sb.append("<p>Further you can check the content of the X.509 certificate with the tool Portecle (<a href=\"http://portecle.sourceforge.net/\">http://portecle.sourceforge.net/</a>). Open the program and select Examine as shown in the following screenshots.<br /><img src=\"/webpage/images/portecle.png\" /><br /><img src=\"/webpage/images/portecle_cert.png\" /></p>\n");
		return sb.toString();
	}

	@Override
	public boolean isProtected() {
		return true;
	}

	@Override
	public String[] getAllowedRoles() {
		return new String[] { UserRole.ADMINISTRATOR.toString(), UserRole.REGISTRATION_USER.toString() };
	}

	@Override
	public Category getCategory() {
		return Category.AUTHENTICATOR;
	}
}