package de.fraunhofer.sit.c2x.pki.ca.module.webserver;

import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.List;

import javax.xml.ws.Endpoint;

import org.apache.log4j.Logger;
import org.eclipse.jetty.http.spi.JettyHttpServer;
import org.eclipse.jetty.http.spi.JettyHttpServerProvider;
import org.eclipse.jetty.http.spi.ThreadPoolExecutorAdapter;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ContextHandlerCollection;
import org.eclipse.jetty.server.ssl.SslSelectChannelConnector;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.ssl.SslContextFactory;

import com.google.inject.Inject;

import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.CallBack;
import de.fraunhofer.sit.c2x.pki.ca.core.interfaces.ThreadPool;
import de.fraunhofer.sit.c2x.pki.ca.core.logging.InjectLogger;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.Servlet;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.ServletWithClientAuth;
import de.fraunhofer.sit.c2x.pki.ca.module.webserver.interfaces.WebServiceWithClientAuth;
import de.fraunhofer.sit.c2x.pki.ca.provider.ProviderException;
import de.fraunhofer.sit.c2x.pki.ca.provider.entities.Authenticator;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.AuthenticatorProvider;
import de.fraunhofer.sit.c2x.pki.ca.provider.interfaces.ConfigProvider;

/**
 * @author Daniel Quanz (daniel.quanz@sit.fraunhofer.de)
 */
public class WebServerWithClientAuth extends WebServer implements CallBack {

	@InjectLogger
	private Logger logger;

	private final List<ServletWithClientAuth> servlets;

	private final WebServiceWithClientAuth service;
	private final AuthenticatorProvider authenticatorProvider;
	private final ConfigProvider configProvider;

	@Inject
	public WebServerWithClientAuth(List<ServletWithClientAuth> servlets, ThreadPool threadpool,
			WebServiceWithClientAuth service, AuthenticatorProvider authenticatorProvider,
			ConfigProvider configProvider) {
		super(threadpool);
		this.servlets = servlets;
		this.service = service;
		this.authenticatorProvider = authenticatorProvider;
		this.configProvider = configProvider;
	}

	@Override
	public void run() {
		// delay startup in milli seconds in order to ensure that the DB is
		// loaded
		try {
			sleep(500);
		} catch (InterruptedException e) {
			logger.error("failed to delay startup of WebserverWithClientAuthentication");
		}
		createAndStartJettyServer();
	}

	@Override
	protected void createAndStartJettyServer() {
		try {
			for (int timeOut = 0; timeOut < 100; timeOut++) {
				if (configProvider.isStarted())
					break;
				if (logger.isTraceEnabled())
					logger.trace("Sleep until config provider is started.");
				sleep(10);
			}
			boolean httpSslEnabled = configProvider.getBoolean("httpSslEnabled", true);
			SslSelectChannelConnector sslSocket = null;
			try {
				sslSocket = getSslSelectChannelConnector();
			} catch (Exception e) {
				logger.error(e);
				httpSslEnabled = false;
			}
			
			if (httpSslEnabled) {
				jettyServer = new Server();
			} else {
				int httpServerPort = configProvider.getInt("httpWebserviceWithClientAuthPort", -1);
				if (httpServerPort == -1)
					throw new IOException(
							"Configuration paramter \"httpWebserviceWithClientAuthPort\" not found in DB. Please update the configuration of the server!");
				jettyServer = new Server(httpServerPort);
			}

			ThreadPoolExecutorAdapter pool = new ThreadPoolExecutorAdapter(threadpool.get());
			jettyServer.setThreadPool(pool);

			JettyHttpServerProvider.setServer(jettyServer);

			ContextHandlerCollection handlers = new ContextHandlerCollection();

			if (httpSslEnabled) {
				jettyServer.addConnector(sslSocket);
			}

			ServletContextHandler context = new ServletContextHandler(ServletContextHandler.SESSIONS);
			context.setContextPath("/");
			for (Servlet sc : servlets) {
				context.addServlet(new ServletHolder(sc), sc.getUrl());
			}
			handlers.addHandler(context);

			jettyServer.setHandler(handlers);

			if (service != null) {
				Endpoint endpoint = Endpoint.create(service);
				JettyHttpServer jettyHttp = new JettyHttpServer(jettyServer, true);
				endpoint.publish(jettyHttp.createContext(service.getPath()));
				logger.info("Web Service is running. " + service.getPath());
			}

			// register call back
			configProvider.registerCallBack("webserverWithClientAuthentication", this);
			configProvider.registerCallBack("httpWebserviceWithClientAuthPort", this);
			configProvider.registerCallBack("httpSslEnabled", this);
			configProvider.registerCallBack("sslServerX509CertFile", this);
			configProvider.registerCallBack("sslServerRsaKeyFile", this);
			configProvider.registerCallBack("sslServerRsaKeyPassword", this);

			if (logger.isInfoEnabled())
				logger.info("Jetty-HttpServer with client authentication is starting ...");
			jettyServer.start();
			jettyServer.join();

		} catch (Exception e) {
			logger.error(e);
		}
	}

	protected SslSelectChannelConnector getSslSelectChannelConnector() throws IOException {
		SslContextFactory sslFactory = new SslContextFactory();
		sslFactory.setProtocol("TLSv1");

		try {
			sslFactory.setKeyStore(super.getKeyStore());
			String password = configProvider.get("sslServerRsaKeyPassword");
			if(password == null || password.isEmpty())
				throw new KeyStoreException("Password \"sslServerRsaKeyPassword\" must not be empty");
			sslFactory.setKeyStorePassword(password);
		} catch (UnrecoverableKeyException | KeyStoreException | NoSuchAlgorithmException
				| CertificateException e) {
			logger.error(e);
			throw new IOException("Unable to create keystore: "+e.getMessage());
		}

		try {
			// add authenticator certificates into trust store
			KeyStore trustStore = KeyStore.getInstance("JKS");
			trustStore.load(null, "".toCharArray());

			Authenticator[] authenticators = authenticatorProvider.getAll(0, Integer.MAX_VALUE);
			if (authenticators != null && authenticators.length > 0) {
				for (Authenticator authenticator : authenticators) {
					trustStore.setCertificateEntry(authenticator.getX509Certificate().getSubjectDN()
							.getName(), authenticator.getX509Certificate());
					if (logger.isDebugEnabled())
						logger.debug(authenticator.getX509Certificate().getType()
								+ " certificate added to trust store: "
								+ authenticator.getX509Certificate().getSubjectDN().getName());
				}
			}
			if (logger.isDebugEnabled())
				logger.debug("Truststore entries after certs from DB added: " + trustStore.size());

			sslFactory.setTrustStore(trustStore);
			sslFactory.setTrustStorePassword("OBF:1twj1ym31fmd1unv1unx1b3g1b4i1uo11uo31fop1ymb1tuj");
		} catch (IOException e) {
			logger.error("Trust store cannot be created.", e);
		} catch (KeyStoreException e) {
			logger.error("Trust store cannot be created.", e);
		} catch (CertificateException e) {
			logger.error("Certificate cannot be created for trust store", e);
		} catch (NoSuchAlgorithmException e) {
			logger.error("Trust store cannot be created.", e);
		} catch (ProviderException e) {
			logger.error("Trust store cannot be created.", e);
		}
		if (configProvider.get("webserverWithClientAuthentication").equals("true")) {
			sslFactory.setNeedClientAuth(true);
			if (logger.isDebugEnabled())
				logger.debug("Jetty-HttpServer with need of client authentication configured");
		} else {
			sslFactory.setNeedClientAuth(false);
			if (logger.isDebugEnabled())
				logger.debug("Jetty-HttpServer without need of client authentication configured");
		}
		SslSelectChannelConnector sslSocket = new SslSelectChannelConnector(sslFactory);

		int httpServerPort = configProvider.getInt("httpWebserviceWithClientAuthPort", -1);
		if (httpServerPort == -1)
			throw new IOException(
					"Configuration paramter \"httpWebserviceWithClientAuthPort\" not found in DB. Please update the configuration of the server!");

		sslSocket.setPort(httpServerPort);
		sslSocket.setConfidentialPort(httpServerPort);
		sslSocket.setForwarded(true);
		return sslSocket;
	}

	public void restartWebServer() {
		// create new thread that restarts the webserver in order to allow the
		// finishing of open processes. Maybe this restart is triggered from a
		// web page
		RestartThread restartThread = new RestartThread(jettyServer);
		restartThread.start();
		// deregister call back
		configProvider.deregisterCallBack("webserverWithClientAuthentication", this);
		configProvider.deregisterCallBack("httpWebserviceWithClientAuthPort", this);
		configProvider.deregisterCallBack("httpSslEnabled", this);
		configProvider.deregisterCallBack("sslServerX509CertFile", this);
		configProvider.deregisterCallBack("sslServerRsaKeyFile", this);
		configProvider.deregisterCallBack("sslServerRsaKeyPassword", this);
	}

	@Override
	public void notifyEvent(String info) {
		// restart the webserver if configuration parameter
		// "webserverWithClientAuthentication" has changed
		if (logger.isInfoEnabled())
			logger.info("Restart of Jetty-HttpServer with client authentication due to change of configuration: "
					+ info);
		restartWebServer();
	}
}
